// $Id: CWindowTools.cpp,v 1.9 2007/02/27 19:47:11 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CWindowTools.hpp"
using Exponent::GUI::Windowing::CWindowTools;


#ifdef WIN32
#include <gdiplus.h>
using namespace Gdiplus;
#else
#include "../Fonts/CFont.hpp"
using Exponent::GUI::Fonts::CFont;
#include <Testing/CTrace.hpp>
using Exponent::Testing::CTrace;
#endif

//	===========================================================================
void CWindowTools::getDesktopWindow(SWindowHandle *handle)
{
#ifdef WIN32
	handle->m_windowHandle = GetDesktopWindow();
#else
	NULL_POINTER(handle->m_windowHandle);
#endif
}

//	===========================================================================
void CWindowTools::getForeGroundWindow(SWindowHandle *handle)
{
#ifdef WIN32
	handle->m_windowHandle = GetForegroundWindow();
#else
	handle->m_windowHandle = ActiveNonFloatingWindow();
#endif
}

//	===========================================================================
void CWindowTools::setForeGroundWindow(SWindowHandle *handle)
{
#ifdef WIN32
	SetForegroundWindow(handle->m_windowHandle);
#else
	BringToFront(handle->m_windowHandle);
#endif
}

//	===========================================================================
void CWindowTools::setFocusWindow(SWindowHandle *handle)
{
#ifdef WIN32
	SetFocus(handle->m_windowHandle);
#else
	SelectWindow(handle->m_windowHandle);
#endif
}

//	===========================================================================
void CWindowTools::getWindowFromPoint(SWindowHandle *handle, const CPoint &point)
{
#ifdef WIN32
	POINT p = { point.getXPosition(), point.getYPosition() };
	handle->m_windowHandle = WindowFromPoint(p);
#else
	Point thePoint = { point.getXPosition(), point.getYPosition() };
	FindWindow(thePoint, &(handle->m_windowHandle));
#endif
}

//	===========================================================================
bool CWindowTools::isWindowVisible(SWindowHandle *handle)
{
	return (IsWindowVisible(handle->m_windowHandle) == TRUE);
}

//	===========================================================================
void CWindowTools::getMousePosition(CPoint &point)
{
#ifdef WIN32
	POINT thePoint;
	GetCursorPos(&thePoint);
	point.setPoint(thePoint.x, thePoint.y);
#else
	Point thePoint;
    GetMouse(&thePoint);
    LocalToGlobal(&thePoint);
	point.setXPosition(thePoint.h);
	point.setYPosition(thePoint.v);
#endif
}

//	===========================================================================
void CWindowTools::setMousePosition(const CPoint &point)
{
#ifdef WIN32
	SetCursorPos(point.getXPosition(), point.getYPosition());
#else
	CGPoint pos = { point.getXPosition(), point.getYPosition() };
    CGDisplayMoveCursorToPoint(CGMainDisplayID(), pos);
#endif
}


//	===========================================================================
bool CWindowTools::computeStringArea(
									#ifndef WIN32
									  const 
									#endif 
									  void *drawContext, const CString &string, CDimension &dimension, void *winFont)
{
	if (drawContext == NULL)
	{
		return false; 
	}
#ifdef WIN32

	// Sme variables for GDI+
	bool hasGdiPlus		   = false;
	ULONG_PTR gdiPlusToken = 0;
	GdiplusStartupInput startupInput;

	// Start GDI+
	if (GdiplusStartup(&gdiPlusToken, &startupInput, NULL) == Ok)
	{
		hasGdiPlus = true;
	}

	bool returnValue = false;

	if (hasGdiPlus)
	{
		// Construct the graphics context
		Gdiplus::Graphics graphics((HDC)drawContext);

		// Convert the string to WCHAR
		const size_t length = strlen(string.getString()) + 1;
		wchar_t *theText    = new wchar_t[length];
		memset(theText, 0, length * sizeof(wchar_t));
		MultiByteToWideChar(CP_ACP, NULL, string.getString(), -1, theText, (int)length);

		// Construct the font object
		Gdiplus::Font theFont((HDC)drawContext, (HFONT)winFont); 

		// Setup the formatting object
		StringFormat format(0);
		format.SetFormatFlags(StringFormatFlagsLineLimit);
		format.SetAlignment(StringAlignmentNear);

		// Setup the position and a bounding rectangle to retrieve the size
		PointF origin(0.0f, 0.0f);
		RectF boundingRectangle;

		//bool returnValue = false;

		// Now try to measure the string
		if (graphics.MeasureString(theText, -1, &theFont, origin, &format, &boundingRectangle) == Ok)
		{
			SizeF theSize;
			boundingRectangle.GetSize(&theSize);
			dimension.setDimension((long)theSize.Width, (long)theSize.Height);
		}
		else
		{
			// Okay we are gonna fall through and try the old way
			SIZE size;
			if (GetTextExtentPoint32((HDC)drawContext, string.getString(), string.getNumberOfCharacters() - 1, &size) == TRUE)
			{
				dimension.setDimension(size.cx, size.cy);
				returnValue = true;
			}
			returnValue = false;
		}

		// We are done with this
		FREE_ARRAY_POINTER(theText);
		//return returnValue;
	}
	else
	{
		SIZE size;
		if (GetTextExtentPoint32((HDC)drawContext, string.getString(), string.getNumberOfCharacters() - 1, &size) == TRUE)
		{
			dimension.setDimension(size.cx, size.cy);
			returnValue = true;
		}
		returnValue = false;
	}
	if (hasGdiPlus)
	{
		GdiplusShutdown(gdiPlusToken);
	}
	return returnValue;
#else
	CFont *font = (CFont*)drawContext;
	
	// First we get the font id from the ATUSI reference
	ATSUFontID fontID;
	const char *fontName = font->getFontName().getString();
	if (ATSUFindFontFromName(fontName, strlen(fontName), kFontFamilyName, kFontNoPlatform, kFontNoScript, kFontNoLanguage, &fontID) != noErr)
	{
		return false;
	}
	
	// Next thing is to create the style object itself
	ATSUStyle style;
	if (ATSUCreateStyle(&style) != noErr)
	{
		return false;
	}
	
	// Next we set the styles attributes
	Fixed theSize       = Long2Fix(font->getFontSize());
	Boolean isBold      = false;		// For Future use
	Boolean isItalic    = false;		// For Future use
	Boolean isUnderline = false;		// For Future use
	
	// Setup the attributes tags and sizes
	ATSUAttributeTag tags[] = { kATSUFontTag, kATSUSizeTag, kATSUQDBoldfaceTag, kATSUQDItalicTag, kATSUQDUnderlineTag };
	ByteCount sizes[]       = { sizeof(ATSUFontID), sizeof(Fixed), sizeof(Boolean), sizeof(Boolean), sizeof(Boolean)  };
	ATSUAttributeValuePtr values[sizeof(tags) / sizeof(ATSUAttributeTag)];

	// Store the values
	values[0] = &fontID;
	values[1] = &theSize;
	values[2] = &isBold;
	values[3] = &isItalic;
	values[4] = &isUnderline;
	
	// Finally we set the attributes in to the style
	if (ATSUSetAttributes(style, sizeof(tags) / sizeof(ATSUAttributeTag), tags, sizes, values) != noErr)
	{
		return false;
	}
	
	// Create the layout object
	ATSUTextLayout layout;
	
	// The string
	CFStringRef stringRef			= string.getStringAsCFStringRef();
	UniCharCount numberOfCharacters = CFStringGetLength(stringRef);
	
	// Can i just say that i dont like all this NewPtr / DisposePtr, its all a bit, well c style malloc like 
	// for me personally! However, this is very localised...
	UniChar *buffer = (UniChar*)NewPtr(numberOfCharacters * sizeof(UniChar));
	CFRange range   = CFRangeMake(0, numberOfCharacters);
	CFStringGetCharacters(stringRef, range, buffer);
	
	// Set the text layout information
	UniCharCount runLengths[1];
	runLengths[0] = numberOfCharacters;
	if (ATSUCreateTextLayoutWithTextPtr(buffer, 0, numberOfCharacters, numberOfCharacters, 1, runLengths, &style, &layout) != noErr)
	{
		return false;
	}
	
	// Now measure the text
	Rect imageRect;
	if (ATSUMeasureTextImage(layout, 0, numberOfCharacters, 0, 0, &imageRect) != noErr)
	{
		return false;
	}
	
	// Now we can delete objects
	DisposePtr((Ptr) buffer);
	ATSUDisposeTextLayout(layout);
	ATSUDisposeStyle(style);

	// Set the dimension
	dimension.setDimension(imageRect.right - imageRect.left, imageRect.bottom - imageRect.top);
	return true;
#endif
}

#ifdef WIN32
//	===========================================================================
bool CWindowTools::computeStringAreaNoGDIPlusStartup(HDC drawContext, const CString &string, CDimension &dimension, HFONT font)
{
	// Construct the graphics context
	Gdiplus::Graphics graphics(drawContext);

	// Convert the string to WCHAR
	const size_t length = strlen(string.getString()) + 1;
	wchar_t *theText    = new wchar_t[length];
	memset(theText, 0, length * sizeof(wchar_t));
	MultiByteToWideChar(CP_ACP, NULL, string.getString(), -1, theText, (int)length);

	// Construct the font object
	Gdiplus::Font theFont(drawContext, font); 

	// Setup the formatting object
	StringFormat format(0);
	format.SetFormatFlags(StringFormatFlagsLineLimit);
	format.SetAlignment(StringAlignmentNear);

	// Setup the position and a bounding rectangle to retrieve the size
	PointF origin(0.0f, 0.0f);
	RectF boundingRectangle;

	bool returnValue = false;

	// Now try to measure the string
	if (graphics.MeasureString(theText, -1, &theFont, origin, &format, &boundingRectangle) == Ok)
	{
		SizeF theSize;
		boundingRectangle.GetSize(&theSize);
		dimension.setDimension((long)theSize.Width, (long)theSize.Height);
		returnValue = true;
	}
	else
	{
		returnValue = false;
	}

	// We are done with this
	FREE_ARRAY_POINTER(theText);
	return returnValue;
}
#endif

//	===========================================================================
bool CWindowTools::getWindowArea(SWindowHandle *handle, CRect &areaToFill)
{
#ifdef WIN32
	RECT winRect;
	if (GetWindowRect(handle->m_windowHandle, &winRect) == TRUE)
	{
		areaToFill.setFromRect(winRect);
		return true;
	}
	return false;
#else
	Rect windBounds;
	GetWindowBounds(handle->m_windowHandle, kWindowStructureRgn, &windBounds);
	areaToFill.setFromRect(windBounds);
	return true;
#endif
}


//	===========================================================================
bool CWindowTools::getContentArea(SWindowHandle *handle, CDimension &dimension)
{
#ifdef WIN32
	RECT winRect;
	if (GetClientRect(handle->m_windowHandle, &winRect) == TRUE)
	{
		dimension.setDimension(winRect.right, winRect.bottom);
		return true;
	}
	return false;
#else
	Rect windBounds;
	GetWindowBounds(handle->m_windowHandle, kWindowContentRgn , &windBounds);
	dimension.setDimension(windBounds.right, windBounds.bottom);
	return true;
#endif
}


//	===========================================================================
bool CWindowTools::computeCorrectedWindowArea(const CDimension &area, CDimension &areaToFill, unsigned long style, unsigned long exStyle, const bool hasMenu)
{
#ifdef WIN32
	RECT winRect = { 0, 0, area.getWidth(), area.getHeight() };
	bool valid   = false;

	if (exStyle == -1)
	{
		if (AdjustWindowRect(&winRect, style, hasMenu) == TRUE)
		{
			valid = true;
		}
	}
	else
	{
		if (AdjustWindowRectEx(&winRect, style, hasMenu, exStyle) == TRUE)
		{
			valid = true;
		}
	}

	if (valid)
	{
		areaToFill.setDimension(winRect.right - winRect.left, winRect.bottom - winRect.top);
	}

	return valid;
#else
	areaToFill = area;
	return true;
#endif
}


//	===========================================================================
bool CWindowTools::getGlobalContentAreaCoordinates(SWindowHandle *windowHandle, CRect &areaToFill)
{
	// Get the window area / position
	CRect windowArea;
	if (!getWindowArea(windowHandle, windowArea))
	{
		return false;
	}

	// Get the client area
	CDimension clientArea;
	if (!getContentArea(windowHandle, clientArea))
	{
		return false;
	}

	const long windowEdgeGap = windowArea.getWidth() - clientArea.getWidth();
	areaToFill.setLeft(windowArea.getLeft() + (windowEdgeGap / 2));

	const long bottomGap = windowArea.getHeight() - (windowEdgeGap / 2);
	const long titleBarHeight = bottomGap - clientArea.getHeight();
	areaToFill.setTop(windowArea.getTop() + titleBarHeight);

	areaToFill.setWidth(clientArea.getWidth());
	areaToFill.setHeight(clientArea.getHeight());

	return true;
}

//	===========================================================================
long CWindowTools::computeLogicalSizeFromPointSize(const long pointSize)
{
#ifdef WIN32
	HDC dc = GetDC(NULL);
	POINT point[2] = {{ 0, 0 }, { 0, GetDeviceCaps(dc, LOGPIXELSY) * pointSize / 72 }};
	DPtoLP(dc, point, 2);
	ReleaseDC(NULL, dc);
	return abs(point[1].y - point[0].y);
#else
	return pointSize;
#endif
}

#ifndef WIN32

//	===========================================================================
bool CWindowTools::isWindowComposited(SWindowHandle *windowHandle)
{
	//CTrace::trace("Checking for compositing window");
	// Check the handle
	if (windowHandle == NULL)
	{
		return false;
	}
	
	//CTrace::trace("Window handle is valid");
	WindowAttributes attributes;
	if (GetWindowAttributes(windowHandle->m_windowHandle, &attributes) == noErr)
	{
		//CTrace::trace("Got window attributes, checking now");
		return ((attributes & kWindowCompositingAttribute) != 0);
	}
	//CTrace::trace("Failed to get window attributes");
	return false;
}
#endif